<?php
/**
 * WooCommerce Compatibility File
 *
 * @link https://woocommerce.com/
 *
 * @package Utech
 */

if ( !class_exists( 'WooCommerce' ) ) {
	return;
}


/**
 * WooCommerce setup function.
 *
 * @link https://docs.woocommerce.com/document/third-party-custom-theme-compatibility/
 * @link https://github.com/woocommerce/woocommerce/wiki/Enabling-product-gallery-features-(zoom,-swipe,-lightbox)
 * @link https://github.com/woocommerce/woocommerce/wiki/Declaring-WooCommerce-support-in-themes
 *
 * @return void
 */
function utech_woocommerce_setup() {
	add_theme_support(
		'woocommerce',
		array(
			'thumbnail_image_width' => 500,
			'single_image_width'    => 800,
			'product_grid'          => array(
				'default_rows'    => 3,
				'min_rows'        => 1,
				'default_columns' => 4,
				'min_columns'     => 1,
				'max_columns'     => 6,
			),
		)
	);
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'utech_woocommerce_setup',100 );

/*--------------------------
	WooCommerce Sidebar Setup
----------------------------*/
function utech_show_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Shop Sidebar', 'utech' ),
		'id'            => 'shop-sidebar',
		'description'   => esc_html__( 'Add widgets here. This widgets will be appear in shop page and when the wocommerce plugin are installed.', 'utech' ),
		'before_widget' => '<div id="%1$s" class="single-widgets %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<div class="widget-title"><h3>',
		'after_title'   => '</h3></div>',
	) );
}
add_action( 'widgets_init', 'utech_show_widgets_init' );


/*-------------------------------
    SHOP PAGE LAYOUT STYLE
--------------------------------*/
if ( ! function_exists( 'utech_shop_layout' ) ) {
    function utech_shop_layout( $type = 'columns' ){
        $sidebar = 'no_sidebar';
        if ( 'full_width' == utech_get_option( 'shop_pages_layout', 'right_sidebar' ) ) {
            $sidebar = 'no_sidebar';
            $utech_columns_layout = 'col-md-12 col-lg-12 col-sm-12 col-xs-12';
        }elseif ( 'left_sidebar' == utech_get_option( 'shop_pages_layout', 'right_sidebar' ) && is_active_sidebar( 'shop-sidebar' ) ) {
            $sidebar = 'left_sidebar';
            $utech_columns_layout = 'col-lg-8 product__content__wrap';
        }elseif ( 'right_sidebar' == utech_get_option( 'shop_pages_layout', 'right_sidebar' ) && is_active_sidebar( 'shop-sidebar' ) ) {
            $sidebar = 'right_sidebar';
            $utech_columns_layout = 'col-lg-8 product__content__wrap';
        }else{
            $utech_columns_layout = 'col-lg-12';
        }

        if ( 'columns' == $type ) {

			if( isset( $_REQUEST['sidebar'] ) && 'no_sidebar' == $_REQUEST['sidebar'] ){
				$utech_columns_layout = 'col-md-12 col-lg-12 col-sm-12 col-xs-12';
			}

            return $utech_columns_layout;
        }

        if ( 'sidebar' == $type ) {

			if( isset( $_REQUEST['sidebar'] ) ){
				$sidebar = $_REQUEST['sidebar'];
			}

            return $sidebar;
        }
    }
}


/**
 * WooCommerce specific scripts & stylesheets.
 *
 * @return void
 */
function utech_woocommerce_scripts() {

	/*------------------------
	ADD STYLES
	--------------------------*/
	wp_enqueue_style( 'utech-woocommerce-style', UTECH_ROOT_URI . '/woocommerce/assets/css/woocommerce.css', array(), UTECH_VERSION );

	$font_path   = WC()->plugin_url() . '/assets/fonts/';
	$inline_font = '@font-face {
		font-family: "star";
		src: url("' . $font_path . 'star.eot");
		src: url("' . $font_path . 'star.eot?#iefix") format("embedded-opentype"),
			url("' . $font_path . 'star.woff") format("woff"),
			url("' . $font_path . 'star.ttf") format("truetype"),
			url("' . $font_path . 'star.svg#star") format("svg");
		font-weight: normal;
		font-style: normal;
	}';

	wp_add_inline_style( 'utech-woocommerce-style', $inline_font );

	/*------------------------
	ADD SCRIPTS
	--------------------------*/
	if( is_product() ){
		wp_enqueue_script('utech-woo', UTECH_ROOT_URI . '/woocommerce/assets/js/woo-active.js',array('jquery'), UTECH_VERSION, true );
	}
}
add_action( 'wp_enqueue_scripts', 'utech_woocommerce_scripts' );


/**
 * Disable the default WooCommerce stylesheet.
 *
 * Removing the default WooCommerce stylesheet and enqueing your own will
 * protect you during WooCommerce core updates.
 *
 * @link https://docs.woocommerce.com/document/disable-the-default-stylesheet/
 */
add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );


/**
 * Trim zeros in price decimals
 **/
add_filter( 'woocommerce_price_trim_zeros', '__return_true' );


/**
 * Add 'woocommerce-active' class to the body tag.
 *
 * @param  array $classes CSS classes applied to the body tag.
 * @return array $classes modified to include 'woocommerce-active' class.
 */
function utech_woocommerce_active_body_class( $classes ) {
	$classes[] = 'woocommerce-active';

	return $classes;
}
add_filter( 'body_class', 'utech_woocommerce_active_body_class' );


/**
 * Change number of products that are displayed per page (shop page)
 */
function utech_shop_products_per_page(  ) {
	// $cols contains the current number of products per page based on the value stored on Options –> Reading
	// Return the number of products you wanna show per page.
	//$cols = utech_get_option( 'product_per_page', 8 );
	return utech_get_option( 'product_per_page', 12 );
}
add_filter( 'loop_shop_per_page', 'utech_shop_products_per_page', 20 );

/**
 * Change number or products per page
 */
function utech_shop_loop_products_per_page() {
	return utech_get_option( 'product_per_page', 12 );
}
add_filter('storefront_products_per_page', 'utech_shop_loop_products_per_page', 999 );

 /**
 * Change number or products per row to 3
 */
add_filter('loop_shop_columns', 'utech_loop_columns', 999);
if (!function_exists('utech_loop_columns')) {
	function utech_loop_columns() {

		$columns = utech_get_option( 'product_per_row', 3 ); // 3 products per row
		if( isset( $_REQUEST['columns'] ) ){
			$columns = $_REQUEST['columns'];
		}
		return $columns;
	}
}

/*-----------------------
DISABLE RELATED PRODUCT
-------------------------*/
if( '0' == utech_get_option( 'enable_related_product', 1 ) ){
	remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_related_products', 20 );
}

/**
 * utech_related_products_heading
 *
 * @param  mixed $content
 * @return string
 */
function utech_related_products_heading( $content ){
	if( '1' == utech_get_option( 'enable_related_product_custom_heading', 0 ) && !empty( utech_get_option( 'related_product_custom_title' ) ) ){
		$content = utech_get_option( 'related_product_custom_title' );
	}
	return $content;
}
add_filter( 'woocommerce_product_related_products_heading', 'utech_related_products_heading');

/**
 * Related Products Args.
 *
 * @param array $args related products args.
 * @return array $args related products args.
 */
function utech_woocommerce_related_products_args( $args ) {
	$defaults = array(
		'posts_per_page' => utech_get_option( 'related_product_per_page', 3 ),
		'columns'        => utech_get_option( 'related_product_per_row', 3 ),
	);
	$args = wp_parse_args( $defaults, $args );
	return $args;
}
add_filter( 'woocommerce_output_related_products_args', 'utech_woocommerce_related_products_args' );

/*---------------------------
    RELATED UPSELLS PRODUCTS SLIDER
-----------------------------*/
if ( !function_exists( 'utech_related_upsells_products_slider' ) ) {
    function utech_related_upsells_products_slider() {

		$related_upsells_product_opt = array(
			'enable_related_slider' => utech_get_option( 'enable_related_product_slider', 1 ),
			'enable_upsells_slider' => utech_get_option( 'enable_upsells_product_slider', 1 ),
		);
		wp_localize_script( 'utech-woo', 'related_upsells_data', $related_upsells_product_opt );
    }
}
add_action( 'wp_enqueue_scripts', 'utech_related_upsells_products_slider' );




/*-----------------------
DISABLE UPSELLS PRODUCT
-------------------------*/
if( '0' == utech_get_option( 'enable_upsells_product', 1 ) ){
	remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_upsell_display', 15 );
}

/**
 * Upsells Products Args.
 *
 * @param array $args upsells products args.
 * @return array $args upsells products args.
 */
function utech_woocommerce_upsell_products_args( $args ) { 

	$args = array(
		'posts_per_page' => utech_get_option( 'upsells_product_per_page', 3 ),
		'columns'        => utech_get_option( 'upsells_product_per_row', 3 ),
	);
    return $args;
};
add_filter( 'woocommerce_upsell_display_args', 'utech_woocommerce_upsell_products_args', 10, 1 );

/**
 * utech_upsells_products_heading
 *
 * @param  mixed $content
 * @return string
 */
function utech_upsells_products_heading( $content ){
	if( '1' == utech_get_option( 'enable_upsells_product_custom_heading', 0 ) && !empty( utech_get_option( 'upsells_product_custom_title' ) ) ){
		$content = utech_get_option( 'upsells_product_custom_title' );
	}
	return $content;
}
add_filter( 'woocommerce_product_upsells_products_heading', 'utech_upsells_products_heading');


/************************************
	PRODUCT MAIN LOOP & SINGLE WRAP
*************************************/
/* ---------------------------------
	REMOVE DEFAULT WooCommerce WRAPPER.
-----------------------------------*/
remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10 );

/* ---------------------------------
	ADD LOOP WRAPPER.
-----------------------------------*/
if ( ! function_exists( 'utech_woocommerce_wrapper_before' ) ) {
	function utech_woocommerce_wrapper_before() {
		?>
			<?php if( is_single() ) : ?>
				<div class="woocommerce__single__product__wrap">
			<?php else : ?>
				<div class="woocommerce__products__wrap">
			<?php endif; ?>
		<?php
	}
}
add_action( 'woocommerce_before_main_content', 'utech_woocommerce_wrapper_before', 10 );

if ( ! function_exists( 'utech_woocommerce_wrapper_after' ) ) {

	function utech_woocommerce_wrapper_after() {
		?>
			</div>
		<?php
	}
}
add_action( 'woocommerce_after_main_content', 'utech_woocommerce_wrapper_after', 10 );


/************************************
	PRODUCT MAIN LOOP
*************************************/
/* ---------------------------------
	REMOVE DEFAULT TOPBAR CONTENT
-----------------------------------*/
remove_action( 'woocommerce_before_main_content', 'woocommerce_breadcrumb', 20 );
remove_action( 'woocommerce_before_shop_loop', 'woocommerce_result_count', 20 );
remove_action( 'woocommerce_before_shop_loop', 'woocommerce_catalog_ordering', 30 );
remove_action( 'woocommerce_before_shop_loop', 'utech_gridlist_toggle_button', 31 );

/*-----------------------------------
	SEPARTE TITLE FORM LINK
------------------------------------*/
remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_product_link_close', 5);
add_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_link_close', 11);

/*-----------------------------------
	ADD LOOP BUTTONS WRAP
-------------------------------------*/

/*-----------------------------------------
	ADD TO CART BUTTON
-----------------------------------------*/
function utech_woocommerce_addcart(){
	echo '<div class="utech__add__to__cart">';
		woocommerce_template_loop_add_to_cart();
	echo '</div>';
}

/*------------------------------------------
    PRODUCT QUICKVIEW BUTTON
-------------------------------------------*/
/**
* [yith_quick_view product_id="30" type="button" label="Quick View"]
* Usages: Compare button shortcode [yith_compare_button] From "YITH WooCommerce Quick View" plugins.
* Plugins URL: https://wordpress.org/plugins/yith-woocommerce-quickview/
* File Path: https://docs.yithemes.com/yith-woocommerce-quick-view/premium-version-settings/shortcode/
* The Function "utech_woocommerce_compare_button" Depends on YITH WooCommerce Compare plugins. If YITH WooCommerce Compare is installed and actived, then it will work.
*/
function utech_quick_view_button( $product_id = 0, $label = '', $return = false ) {
    if( !class_exists('YITH_WCQV_Frontend') ){
        return;
    }
    global $product;

    if( ! $product_id ){
        $product instanceof WC_Product && $product_id = yit_get_prop( $product, 'id', true );
    }
    $show_quick_view_button = apply_filters( 'yith_wcqv_show_quick_view_button', true, $product_id );
    if( !$show_quick_view_button ) return;

    $button = '';
    if( $product_id ) {
        // get label
        $label  = $label ? $label : '';
        $button = '<div class="utech__quickview__button"><a title="'.esc_attr__( 'Quick View', 'utech' ).'" href="#" class="button yith-wcqv-button" data-product_id="' . $product_id . '"><i class="ti ti-zoom-in"></i>' . $label . '</a></div>';
        $button = apply_filters('yith_add_quick_view_button_html', $button, $label, $product);
    }
    if( $return ) {
        return $button;
    }
    echo wp_kses_post( $button );
}
remove_action( 'woocommerce_after_shop_loop_item', 'yith_add_quick_view_button', 15 );
remove_action( 'yith_wcwl_table_after_product_name', 'yith_add_quick_view_button', 15 );

/*------------------------------------------
    PRODUCT WISHLIST BUTTON
-------------------------------------------*/
/**
* Usages: "utech_add_to_wishlist_button()" function is used  to modify the wishlist button from "YITH WooCommerce Wishlist" plugins.
* Plugins URL: https://wordpress.org/plugins/yith-woocommerce-wishlist/
* File Path: yith-woocommerce-wishlist/templates/add-to-wishlist.php
* The below Function depends on YITH WooCommerce Wishlist plugins. If YITH WooCommerce Wishlist is installed and actived, then it will work.
*/
function utech_add_to_wishlist_button( $normalicon = '<i class="fa fa-heart-o"></i>', $addedicon = '<i class="fa fa-heart"></i>', $tooltip = 'no' ) {
    global $product, $yith_wcwl;

    if ( ! class_exists( 'YITH_WCWL' ) || empty(get_option( 'yith_wcwl_wishlist_page_id' ))) return;

    $url          = YITH_WCWL()->get_wishlist_url();
    $product_type = $product->get_type();
    $exists       = $yith_wcwl->is_product_in_wishlist( $product->get_id() );
    $classes      = 'class="add_to_wishlist"';
    $add          = get_option( 'yith_wcwl_add_to_wishlist_text' );
    $browse       = get_option( 'yith_wcwl_browse_wishlist_text' );
    $added        = get_option( 'yith_wcwl_product_added_text' );

    $output = '';
    $output  .= '<div class="'.( $tooltip == 'yes' ? '' : 'tooltip_no' ).' wishlist button-default yith-wcwl-add-to-wishlist add-to-wishlist-' . esc_attr( $product->get_id() ) . '">';
        $output .= '<div class="yith-wcwl-add-button';
            $output .= $exists ? ' hide" style="display:none;"' : ' show"';
            $output .= '><a href="' . esc_url( htmlspecialchars( YITH_WCWL()->get_wishlist_url() ) ) . '" data-product-id="' . esc_attr( $product->get_id() ) . '" data-product-type="' . esc_attr( $product_type ) . '" ' . $classes . ' >'.$normalicon.'<span class="utech__product__action__tooltip">'.esc_html( $add ).'</span></a>';
            $output .= '<i class="fa fa-spinner fa-pulse ajax-loading" style="visibility:hidden"></i>';
        $output .= '</div>';

        $output .= '<div class="yith-wcwl-wishlistaddedbrowse show" style="display:block;"><a class="" href="' . esc_url( $url ) . '">'.$addedicon.'<span class="utech__product__action__tooltip">'.esc_html( $browse ).'</span></a></div>';
        $output .= '<div class="yith-wcwl-wishlistexistsbrowse ' . ( $exists ? 'show' : 'hide' ) . '" style="display:' . ( $exists ? 'block' : 'none' ) . '"><a href="' . esc_url( $url ) . '" class="">'.$addedicon.'<span class="utech__product__action__tooltip">'.esc_html( $added ).'</span></a></div>';
    $output .= '</div>';
    echo wp_kses_post($output);
}

/*------------------------------------------
    PRODUCT COMPARE BUTTON
-------------------------------------------*/
/**
* Usages: Compare button shortcode [yith_compare_button] From "YITH WooCommerce Compare" plugins.
* Plugins URL: https://wordpress.org/plugins/yith-woocommerce-compare/
* File Path: yith-woocommerce-compare/includes/class.yith-woocompare-frontend.php
* The Function "utech_woocommerce_compare_button" Depends on YITH WooCommerce Compare plugins. If YITH WooCommerce Compare is installed and actived, then it will work.
*/
function utech_woocommerce_compare_button( $buttonstyle = 1, $compare_icon = '<i class="ti ti-reload"></i>', $label = '' ){

    if( !class_exists( 'YITH_Woocompare' ) ) {
        return;
    }
    global $product;

    $product_id = $product->get_id();
    $comp_link  = site_url() . '?action=yith-woocompare-add-product';
    $comp_link  = add_query_arg('id', $product_id, $comp_link);

	$label = $label ? $label : esc_html__( 'Compare', 'utech' );

    if( $buttonstyle == 1 ){
        echo do_shortcode('[yith_compare_button]');
    }else{
        echo '<a href="'. esc_url( $comp_link ) .'" class="utech__compare__button woocommerce product compare-button compare" data-product_id="'. esc_attr( $product_id ) .'" rel="nofollow">'.$compare_icon.$label.'</a>';
    }
}

function utech_loop_product_buttons(){
	?>
	<div class="woocommerce__loop__product__action__buttons">
		<?php
			utech_woocommerce_addcart();
			utech_quick_view_button();
			utech_add_to_wishlist_button();
			utech_woocommerce_compare_button();
		?>
	</div>
	<?php
}

if( '1' == utech_get_option( 'enable_product_group_buttons', 1 ) ){
	add_action( 'woocommerce_before_shop_loop_item_title', 'utech_loop_product_buttons', 12 );
}

/*----------------------------------
	ADD LINK TO TITLE
-----------------------------------*/
remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );
function utech_product_loop_title() {

    echo '<h3 class="' . esc_attr( apply_filters( 'woocommerce_product_loop_title_classes', 'woocommerce__loop__product__title' ) ) . '">' . '<a href="' . esc_url( get_the_permalink() ) . '">' . get_the_title() . '</a>' . '</h3>';

}
add_action( 'woocommerce_shop_loop_item_title', 'utech_product_loop_title', 10 );

/*----------------------------------
	ADD EXCERPT TO LOOP CONTENT
-----------------------------------*/
add_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_single_excerpt', 5);


/* ----------------------------------
	LOOP ITEM WRAP
-----------------------------------*/
if( !function_exists('utech_before_loop_product_item_wrap') ){
	function utech_before_loop_product_item_wrap(){
		echo '<div class="woocommerce__loop__product__item">';
	}
}
add_action( 'woocommerce_before_shop_loop_item', 'utech_before_loop_product_item_wrap', 0 );

if( !function_exists('utech_after_loop_product_item_wrap') ){
	function utech_after_loop_product_item_wrap(){
		echo '</div>';
	}
}
add_action( 'woocommerce_after_shop_loop_item', 'utech_after_loop_product_item_wrap', 30 );


/* ----------------------------------
	LOOP THUMBNAIL WRAP
-----------------------------------*/
if( !function_exists('utech_before_loop_product_thumbnail_wrap') ){
	function utech_before_loop_product_thumbnail_wrap(){
		echo '<div class="woocommerce__loop__product__thumbnail">';
	}
}
add_action( 'woocommerce_before_shop_loop_item', 'utech_before_loop_product_thumbnail_wrap', 9 );

if( !function_exists('utech_after_loop_product_thumbnail_wrap') ){
	function utech_after_loop_product_thumbnail_wrap(){
		echo '</div>';
	}
}
add_action( 'woocommerce_before_shop_loop_item_title', 'utech_after_loop_product_thumbnail_wrap', 12 );

/* ----------------------------------
	LOOP PRODUCT CONTENT WRAP
-----------------------------------*/
if( !function_exists('utech_before_loop_product_content_wrap') ){
	function utech_before_loop_product_content_wrap(){
		echo '<div class="woocommerce__loop__product__content">';
	}
}
add_action( 'woocommerce_shop_loop_item_title', 'utech_before_loop_product_content_wrap', 9 );

if( !function_exists('utech_after_loop_product_content_wrap') ){
	function utech_after_loop_product_content_wrap(){
		echo '</div>';
	}
}
add_action( 'woocommerce_after_shop_loop_item', 'utech_after_loop_product_content_wrap', 11 );


/* ----------------------------------
	LOOP PRICE & RATING WRAP
-----------------------------------*/
remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 5 );
//add_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 10 );
remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_price', 10 );
add_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_price', 5 );
if( !function_exists('utech_before_loop_product_price_rating_wrap') ){
	function utech_before_loop_product_price_rating_wrap(){
		echo '<div class="woocommerce__loop__price__rating">';
	}
}
add_action( 'woocommerce_after_shop_loop_item_title', 'utech_before_loop_product_price_rating_wrap', 0 );

if( !function_exists('utech_after_loop_product_price_rating_wrap') ){
	function utech_after_loop_product_price_rating_wrap(){
		echo '</div>';
	}
}
add_action( 'woocommerce_after_shop_loop_item_title', 'utech_after_loop_product_price_rating_wrap', 30 );


/* --------------------------------
REMOVE LOOP DEFAULT CART BUTTON
----------------------------------*/
function utech_shop_woocommerce_active_body_class( $classes ) {
	$classes[] = 'default__add__to__cart__button';
	return $classes;
}
if( '1' == utech_get_option( 'enable_add_to_cart_default_button', 0 ) ){
	add_filter( 'body_class', 'utech_shop_woocommerce_active_body_class' );
}else{
	remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
}

/* --------------------------------
	ADD PRODUCT DETAILS BUTTON
----------------------------------*/
function utech_get_single_product_link(){
	echo '<a href="'.get_the_permalink().'" class="single__product__view__link">'.utech_get_option( 'product_view_in_details_button_text', esc_html__('View Details', 'utech' ) ).'</a>';
}

if( '1' == utech_get_option( 'enable_product_view_in_details_button', 0 ) ){
	add_action( 'woocommerce_after_shop_loop_item', 'utech_get_single_product_link', 10 );
}

/* --------------------------------
	WOOCOMMERCE PAGINATION
----------------------------------*/
remove_action( 'woocommerce_after_shop_loop', 'woocommerce_pagination', 10 );

if ( ! function_exists( 'utech_woocommerce_pagination' ) ) {

	function utech_woocommerce_pagination() {

		if( '0' == utech_get_option( 'show_shop_pagination', 1 ) ){
			return;
		}

		?>
			<div class="post-pagination">                        
				<?php utech_pagination(); ?>
			</div>
		<?php
	}
}
add_action( 'woocommerce_after_shop_loop', 'utech_woocommerce_pagination', 10 );


function show_rating_in_shop_loop_products() {

    global $product;
    $rating = $product->get_average_rating();
    echo '<div class="star-rating ehi-star-rating"><span style="width:' . esc_attr(( $rating / 5 ) * 100) . '%"></span></div>';
}
add_action('woocommerce_after_shop_loop_item_title', 'show_rating_in_shop_loop_products', 10 );


/************************************
	SIGNLE PRODUCT CHANGE
*************************************/
/* ---------------------------------
	SINGLE PRODUCT PAGE STYLE CHANGE
-----------------------------------*/

//remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );
if( '1' == utech_get_option( 'desable_product_title_from_summary', 0 ) ){
	remove_action('woocommerce_single_product_summary', 'woocommerce_template_single_title', 5 );
}

function utech_single_product_button_wrap_start(){
	echo '<div class="single__product__details__wrap">';
}
add_action('woocommerce_single_product_summary','utech_single_product_button_wrap_start',0);

function utech_single_product_button_wrap_end(){
	echo '</div>';
}
add_action('woocommerce_single_product_summary','utech_single_product_button_wrap_end',35);



/**
 * Sample implementation of the WooCommerce Mini Cart.
 *
 * You can add the WooCommerce Mini Cart to header.php like so ...
 *
	<?php
		if ( function_exists( 'utech_woocommerce_header_cart' ) ) {
			utech_woocommerce_header_cart();
		}
	?>
 */

if ( ! function_exists( 'utech_woocommerce_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments.
	 *
	 * Ensure cart contents update when products are added to the cart via AJAX.
	 *
	 * @param array $fragments Fragments to refresh via AJAX.
	 * @return array Fragments to refresh via AJAX.
	 */
	function utech_woocommerce_cart_link_fragment( $fragments ) {
		ob_start();
		utech_woocommerce_cart_link();
		$fragments['a.cart-contents'] = ob_get_clean();

		return $fragments;
	}
}
add_filter( 'woocommerce_add_to_cart_fragments', 'utech_woocommerce_cart_link_fragment' );

if ( ! function_exists( 'utech_woocommerce_cart_link' ) ) {
	/**
	 * Cart Link.
	 *
	 * Displayed a link to the cart including the number of items present and the cart total.
	 *
	 * @return void
	 */
	function utech_woocommerce_cart_link() {
		?>

		<a class="cart-contents cart-button" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'utech' ); ?>">
			<div class="total__cart__item">
				<i class="ti-shopping-cart"></i>
				<span class="cart-count"><?php echo esc_html( WC()->cart->get_cart_contents_count() ); ?></span>
			</div>
			<span class="quick__link__text"><?php echo utech_get_option( 'quick_actions_cart_link_text' , esc_html__( 'Cart', 'utech' ) ); ?></span>
		</a>

		<?php
	}
}

if ( ! function_exists( 'utech_woocommerce_header_cart' ) ) {
	/**
	 * Display Header Cart.
	 *
	 * @return void
	 */
	function utech_woocommerce_header_cart() { ?>


		<div class="action__cart__button__and__item">
			<?php utech_woocommerce_cart_link(); ?>
			<?php if (  !is_cart() && !is_checkout() ) : ?>
				<div class="header__mini__cart__content">
					<?php
						$instance = array(
							'title' => '',
						);

						the_widget( 'WC_Widget_Cart', $instance );
					?>
				</div>
			<?php endif; ?>
		</div>

		<?php
	}
}

/**
 * Add Custom Plus and Minus Button On WooCommerce Quantity.
 */
if ( !function_exists('utech_quantity_fields_script') ) {
	function utech_quantity_fields_script(){
	    ?>
	    <script type='text/javascript'>
	    jQuery( function( $ ) {
	        if ( ! String.prototype.getDecimals ) {
	            String.prototype.getDecimals = function() {
	                var num = this,
	                    match = ('' + num).match(/(?:\.(\d+))?(?:[eE]([+-]?\d+))?$/);
	                if ( ! match ) {
	                    return 0;
	                }
	                return Math.max( 0, ( match[1] ? match[1].length : 0 ) - ( match[2] ? +match[2] : 0 ) );
	            }
	        }
	        // Quantity "plus" and "minus" buttons
	        $( document.body ).on( 'click', '.plus, .minus', function() {
	            var $qty        = $( this ).closest( '.quantity' ).find( '.qty'),
	                currentVal  = parseFloat( $qty.val() ),
	                max         = parseFloat( $qty.attr( 'max' ) ),
	                min         = parseFloat( $qty.attr( 'min' ) ),
	                step        = $qty.attr( 'step' );

	            // Format values
	            if ( ! currentVal || currentVal === '' || currentVal === 'NaN' ) currentVal = 0;
	            if ( max === '' || max === 'NaN' ) max = '';
	            if ( min === '' || min === 'NaN' ) min = 0;
	            if ( step === 'any' || step === '' || step === undefined || parseFloat( step ) === 'NaN' ) step = 1;

	            // Change the value
	            if ( $( this ).is( '.plus' ) ) {
	                if ( max && ( currentVal >= max ) ) {
	                    $qty.val( max );
	                } else {
	                    $qty.val( ( currentVal + parseFloat( step )).toFixed( step.getDecimals() ) );
	                }
	            } else {
	                if ( min && ( currentVal <= min ) ) {
	                    $qty.val( min );
	                } else if ( currentVal > 0 ) {
	                    $qty.val( ( currentVal - parseFloat( step )).toFixed( step.getDecimals() ) );
	                }
	            }

	            // Trigger change event
	            $qty.trigger( 'change' );

	            return false;
	        });
	    });
	    </script>
	    <?php
	}
}
add_action( 'wp_footer' , 'utech_quantity_fields_script' );

/* ------------------------------
	Added Checkout Button Script 
-------------------------------*/
function order_checkout_jquery_script() {
  if ( is_checkout() && ! is_wc_endpoint_url() ) : ?>
	<script type="text/javascript">
		jQuery( function($){
			jQuery('form.checkout').on('submit', function(event) {
				jQuery('button#place_order').text('Please Wait');
				event.preventDefault();
			});
		});
	</script>
 	<?php
  endif;
}
add_action( 'wp_footer', 'order_checkout_jquery_script' );
add_filter( 'woocommerce_defer_transactional_emails','__return_true' );

/* ------------------------------
	WOO GRID LIST TOGGLE
-------------------------------*/
require_once( dirname(__FILE__) . '/woo-grid-list.php' );


/* -----------------------------
	WOO PERCENTAGE SALE BADGE
-------------------------------*/
function utech_add_percentage_to_sale_badge( $html, $post, $product ) {

    if ( $product->is_type( 'variable' ) ) {
        $percentages = array();

        // Get all variation prices
        $prices = $product->get_variation_prices();

        // Loop through variation prices
        foreach ( $prices['price'] as $key => $price ) {
            // Only on sale variations
            if ( $prices['regular_price'][$key] !== $price ) {
                // Calculate and set in the array the percentage for each variation on sale
                $percentages[] = round( 100 - ( floatval( $prices['sale_price'][$key] ) / floatval( $prices['regular_price'][$key] ) * 100 ) );
            }
        }
        // We keep the highest value
        $percentage = max( $percentages ) . '%';

    } elseif ( $product->is_type( 'grouped' ) ) {
        $percentages = array();

        // Get all variation prices
        $children_ids = $product->get_children();

        // Loop through variation prices
        foreach ( $children_ids as $child_id ) {
            $child_product = wc_get_product( $child_id );

            $regular_price = (float) $child_product->get_regular_price();
            $sale_price    = (float) $child_product->get_sale_price();

            if ( $sale_price != 0 || !empty( $sale_price ) ) {
                // Calculate and set in the array the percentage for each child on sale
                $percentages[] = round( 100 - ( $sale_price / $regular_price * 100 ) );
            }
        }
        // We keep the highest value
        $percentage = max( $percentages ) . '%';

    } else {
        $regular_price = (float) $product->get_regular_price();
        $sale_price    = (float) $product->get_sale_price();

        if ( $sale_price != 0 || !empty( $sale_price ) ) {
            $percentage = round( 100 - ( $sale_price / $regular_price * 100 ) ) . '%';
        } else {
            return $html;
        }
    }
    return '<span class="onsale">' .  $percentage . ' ' . esc_html__( 'OFF', 'utech' ) . '</span>';
}

if( '1' == utech_get_option( 'enable_percentage_bedge', 1 ) ){
	add_filter( 'woocommerce_sale_flash', 'utech_add_percentage_to_sale_badge', 20, 3 );
}


/*===========================
ONE PAGE CHECKOUT
============================*/
// https://rudrastyh.com/woocommerce/redirect-to-checkout-skip-cart.html
// https://yourblogcoach.com/how-to-redirect-to-checkout-after-add-to-cart/
// https://wordpress.org/plugins/woocommerce-direct-checkout/

/*--------------------------
REDIRECT TO CHECKOUT URL
----------------------------*/
function utech_redirect_addcart_to_checkout() {
	global $woocommerce;
	$checkout_url = $woocommerce->cart->get_checkout_url();
	return $checkout_url;
}
if( '1' == utech_get_option('redirect_one_page_checkout', 0 ) ){
	add_filter( 'woocommerce_add_to_cart_redirect', 'utech_redirect_addcart_to_checkout' );
	add_filter('option_woocommerce_enable_ajax_add_to_cart', '__return_false');
	add_filter('option_woocommerce_cart_redirect_after_add', '__return_false');
}


/*-----------------------------------
CHANGE BUTTON TEXT ARCHIVE PAGES
-----------------------------------*/
function utech_add_to_cart_text_1( $add_to_cart_html ) {
	return str_replace( 'Add to cart', utech_get_option('custom_add_to_cart_button_text', 'Buy now' ), $add_to_cart_html );
}
if( '1' == utech_get_option('change_add_to_cart_button_text', 0 ) ){
	add_filter( 'woocommerce_loop_add_to_cart_link', 'utech_add_to_cart_text_1' );
}

/*-----------------------------------
CHANGE BUTTON TEXT PRODUCT PAGES
-----------------------------------*/
function utech_add_to_cart_text_2( $product ){
	return utech_get_option('cart_button_text_single_product', 'Buy now' );
}

if( '1' == utech_get_option('change_cart_button_text_single_product', 0 ) ){
	add_filter( 'woocommerce_product_single_add_to_cart_text', 'utech_add_to_cart_text_2' );
}

/*-----------------------------------
CHANGE VRIABLE PRODUCT BUTTON TEXT
-----------------------------------*/
function utech_variable_product_custom_cart_button_text( $button_text, $product ) {
    if ( $product->is_type( 'variable' ) )
        $button_text = utech_get_option('variable_product_custom_cart_button_text', 'Buy now' );
    return $button_text;
}
if( '1' == utech_get_option('change_variable_product_custom_cart_button_text', 0 ) ){
	add_filter( 'woocommerce_product_add_to_cart_text', 'utech_variable_product_custom_cart_button_text', 10, 2 );
}

/*-----------------------------------
CHANGE VRIABLE PRODUCT BUTTON TEXT SINGLE PRODUCT
-----------------------------------*/
function utech_variable_single_product_custom_cart_button_text( $button_text, $product ) {
    if ( $product->is_type( 'variable' ) ){
        $button_text = utech_get_option('variable_single_product_custom_cart_button_text', 'Buy now' );
	}
    return $button_text;
}
if( '1' == utech_get_option('change_variable_single_product_custom_cart_button_text', 0 ) ){
	add_filter( 'woocommerce_product_single_add_to_cart_text', 'utech_variable_single_product_custom_cart_button_text', 10, 2 );
}

/*--------------------------
REMOVE ADDED TO CART MESSAGE
----------------------------*/
function utech_remove_add_to_cart_success_message( $message ) {
	return '';
}
if( '1' == utech_get_option('remove_add_to_cart_success_message', 0 ) ){
	add_filter( 'wc_add_to_cart_message_html', 'utech_remove_add_to_cart_success_message' );
}

/*--------------------------
REMOVE CUPON FROM CHECKOUT PAGE
----------------------------*/
if( '1' == utech_get_option('remove_checkout_page_cupon', 0 ) ){
	remove_action( 'woocommerce_before_checkout_form', 'woocommerce_checkout_coupon_form', 10 ); 
}

/*---------------------------
ADD NEW FIELD IN CHECKOUT FORM
-----------------------------*/
function utech_checkout_field_extra_data(  $checkout ){
	
	woocommerce_form_field( 'full_name', array(
		'type'        => 'text',
		'required'    => true,
		'label'       => esc_html__( 'Your full name', 'utech' ),
		'priority' => 10,
	), $checkout->get_value( 'full_name' ) );
	
}
//add_action( 'woocommerce_after_checkout_billing_form', 'utech_checkout_field_extra_data' );
//add_action( 'woocommerce_after_checkout_shipping_form', 'utech_checkout_field_extra_data' );

/*---------------------------
SAVED NEW FIELD VALUE CHECKOUT FORM
-----------------------------*/
function utech_save_checkout_field_added_data( $order_id ){

	if( !empty( $_POST['full_name'] ) ){
		update_post_meta( $order_id, 'full_name', sanitize_text_field( $_POST['full_name'] ) );
	}
}
//add_action( 'woocommerce_checkout_update_order_meta', 'utech_save_checkout_field_added_data' );


/*---------------------------
REMOVE CHECKOUT FORM FIELDS
-----------------------------*/
function utech_remove_checkout_fields( $fields ) {

	if ( $remove = utech_get_option( 'remove_checkout_fields', array() ) ) {

		foreach ( $remove as $id => $key ) {
			// We need to remove both fields otherwise will be required If need any key just try with continue. example: if ( $key == 'country' ) { continue; }.
			unset( $fields['billing']['billing_' . $key] );
			unset( $fields['shipping']['shipping_' . $key] );
		}
	}
	return $fields;
}
add_filter( 'woocommerce_checkout_fields', 'utech_remove_checkout_fields' );


/*-----------------------------
LOGIN WITH PHONE NUMBER
------------------------------*/
function utech_login_with_phone_number($user, $username, $password) {
	
	//  Try logging in via their billing phone number
	if (is_numeric($username)) {

		//  The passed username is numeric - that's a start
		//  Now let's grab all matching users with the same phone number:

		$matchingUsers = get_users(array(
			'meta_key'     => 'billing_phone',
			'meta_value'   => $username,
			'meta_compare' => 'LIKE'
		));

		//  Let's save time and assume there's only one.
		if (is_array($matchingUsers) && !empty($matchingUsers)) {
			$username = $matchingUsers[0]->user_login;
		}

	}elseif (is_email($username)) {

		//  The passed username is email- that's a start
		//  Now let's grab all matching users with the same email:

		$matchingUsers = get_user_by('email', $username);

		//  Let's save time and assume there's only one.
		if (isset($matchingUsers->user_login)) {
			$username = $matchingUsers->user_login;
		}

	}

	return wp_authenticate_username_password(null, $username, $password);
}

if ( '1' == utech_get_option( 'logn_with_phone_number', 0 ) ) {
	add_filter('authenticate', 'utech_login_with_phone_number', 20, 3);
}

/*-----------------------------
EDIT FORM CORE FIELD VALUE
------------------------------*/
function custom_override_checkout_fields( $fields ) {
	
	if ( '1' == utech_get_option( 'change_checkout_form_first_name_label', 0 ) ) {
		$fields['billing']['billing_first_name']['label'] = utech_get_option( 'checkout_form_first_name_label','Your Full Name' );
	}
	if ( '1' == utech_get_option( 'change_checkout_form_address_1_label', 0 ) ) {
		$fields['billing']['billing_address_1']['label']  = utech_get_option( 'checkout_form_address_1_label','Delivary Address' );
	}
	if ( '1' == utech_get_option( 'change_checkout_form_phone_label', 0 ) ) {
		$fields['billing']['billing_phone']['label']  = utech_get_option( 'checkout_form_phone_label','Your Phone Number' );
	}
	return $fields;
}
add_filter( 'woocommerce_checkout_fields' , 'custom_override_checkout_fields' );

/*---------------------------
PLACE ORDER BUTTON TEXT
-----------------------------*/
function utech_custom_order_button_text( $button_text ) {
	return utech_get_option( 'place_order_button_text', 'Place Order' );
}
if ( '1' == utech_get_option( 'change_place_order_button_text', 0 ) ) {
	add_filter( 'woocommerce_order_button_text', 'utech_custom_order_button_text' );
}

/*---------------------------
REMOVE CHECKOUT SHIPING ADDRESS
-----------------------------*/
function utech_remove_checkout_shipping_address( $val ){

	if ( '1' == utech_get_option( 'remove_checkout_shipping_address', 0 ) ) {
		$val = 'billing_only';
	}
	return $val;
}
add_filter('option_woocommerce_ship_to_destination', 'utech_remove_checkout_shipping_address', 10, 3);

/*---------------------------
REMOVE CHECKOUT ORDER NOTES
-----------------------------*/
if( '1' == utech_get_option('remove_checkout_order_notes_fields', 0 ) ){
	add_filter( 'woocommerce_enable_order_notes_field', '__return_false' );
}

/*---------------------------
REMOVE CHECKOUT POLICY TEXT
-----------------------------*/
if ( '1' == utech_get_option( 'remove_checkout_privacy_policy_text', 0 ) ) {
	remove_action( 'woocommerce_checkout_terms_and_conditions', 'wc_checkout_privacy_policy_text', 20 );
}

/*---------------------------
REMOVE CHECKOUT TERMS & CONDI
-----------------------------*/
if ( '1' == utech_get_option( 'remove_checkout_terms_and_conditions', 0 ) ) {
	add_filter( 'woocommerce_checkout_show_terms', '__return_false' );
	remove_action( 'woocommerce_checkout_terms_and_conditions', 'wc_terms_and_conditions_page_content', 30 );
}

/* ----------------------------------
	CHANGE SHIPPING METHOD HEADING
------------------------------------*/
function utech_shipping_package_name(){
	if( is_checkout() ){
		return '<h3>'.utech_get_option( 'checkout_shipping_heading', 'Shipping' ).'</h3>';
	}else{
		return utech_get_option( 'checkout_shipping_heading', 'Shipping' );
	}
}
add_filter( 'woocommerce_shipping_package_name', 'utech_shipping_package_name' );

/* -----------------------------
	REMOVE PRODUCT FORM CHECKOUT
-------------------------------*/
function utech_cart_item_remove_link( $product_name, $cart_item, $cart_item_key ) {

    if ( is_checkout() ) {

        $product_name .= apply_filters( 'woocommerce_cart_item_remove_link', sprintf(
            '<a href="%s" rel="nofollow" class="remove-icon" style="float:left;">x</a>',
            esc_url( wc_get_cart_remove_url( $cart_item_key ) ),
            __( 'Remove this item', 'utech' ),
            esc_attr( $cart_item['product_id'] ),
            esc_attr( $cart_item['data']->get_sku() )
        ), $cart_item_key );

        return $product_name;
    }

}
if ( '1' == utech_get_option( 'remove_product_from_checkout', 0 ) && is_checkout() ) {
	add_filter( 'woocommerce_cart_item_name', 'utech_cart_item_remove_link', 10, 3 );
}

/* -----------------------------
	DISABLE SHIPPING CALCULATE ON CART PAGE
-------------------------------*/
function utech_disable_shipping_calc_on_cart( $show_shipping ) {
	if( is_cart() ) {
		return false;
	}
	return $show_shipping;
}
if ( '1' == utech_get_option( 'disable_shipping_calc_on_cart', 0 ) ) {
	add_filter( 'woocommerce_cart_ready_to_calc_shipping', 'utech_disable_shipping_calc_on_cart', 99 );
}

/*===================================
	PRODUCT CUSTOMIZE
=====================================*/
function utech_add_bdt_text_currency( $cw_currency ) {
	$cw_currency['woo_custom_currency'] = __( 'Custom Currency', 'utech' );
	return $cw_currency;
}
if ( '1' == utech_get_option( 'product_custom_currency_text_or_sign', 0 ) ) {
	add_filter( 'woocommerce_currencies', 'utech_add_bdt_text_currency' );
}
function utech_add_bdt_text_currency_symbol( $custom_currency_symbol, $custom_currency ) {
	switch( $custom_currency ) {
		case 'woo_custom_currency': $custom_currency_symbol = utech_get_option( 'product_custom_currency_text', 'BDT' );
		break;
	}
	return $custom_currency_symbol;
}
if ( '1' == utech_get_option( 'product_custom_currency_text_or_sign', 0 ) ) {
	add_filter('woocommerce_currency_symbol', 'utech_add_bdt_text_currency_symbol', 10, 2);
}

/* ----------------------------
	MOBILE QUICKLINK
------------------------------*/

global $quick_links_data;
$quick_links_data = [];
$quick_links_data['home_link'] = '
	<li>
		<a href="' . esc_url( home_url( '/' ) ) . '" class="homepage__link">
			<i class="ti-home"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_home_link_text' , esc_html__( 'Home', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['cart_link'] = '
	<li>
		<a href="' . esc_url( wc_get_cart_url() ) . '" class="cart__link cart-contents">
			<i class="ti-shopping-cart"><span class="cart-count">0</span></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_cart_link_text' , esc_html__( 'Cart', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['checkout_link'] = '
	<li>
		<a href="' . esc_url( wc_get_checkout_url() ) . '" class="checkout__link">
			<i class="ti-credit-card"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_checkout_link_text' , esc_html__( 'Checkout', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['compare_link'] = '
	<li>
		<a href="#" class="compare__product">
			<i class="ti-reload"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_compare_link_text' , esc_html__( 'Compare', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['recent_view_link'] = '
	<li>
		<a href="#" class="recently__view">
			<i class="ti-time"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_recent_view_link_text' , esc_html__( 'Viewed', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['wishlist_link'] = '
	<li>
		<a href="#" class="wishlist__link">
			<i class="ti-heart"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_wishlist_link_text' , esc_html__( 'Wishlist', 'utech' ) ) . '</span>
		</a>
	</li>
';

$quick_links_data['account_link'] = '
	<li>
		<a href="' . esc_url( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . '" class="user__account">
			<i class="ti-user"></i>
			<span class="quick__link__text">' . utech_get_option( 'quick_actions_wishlist_link_text' , esc_html__( 'Account', 'utech' ) ) . '</span>
		</a>
	</li>
';

global $active_quick_actions_links;
$active_quick_actions_links = utech_get_option( 'mobile_quick_actions_links', array() );

/* ------------------
	DEFAULT SERIAL
----------------------*/
// foreach( $quick_links_data as $single_link => $single_link_data ){
// 	if( in_array( $single_link, $active_quick_actions_links ) ){ 
// 		echo utech_kses($single_link_data);
// 	}
// }

/* ------------------
	SHORTING SERIAL
----------------------*/
// foreach( $active_quick_actions_links as $single_link => $single_link_data ){
// 	echo utech_kses($quick_links_data[$single_link_data]);
// }

function utech_mobile_quick_action_menu(){
	global $quick_links_data,$active_quick_actions_links;
	if( !$active_quick_actions_links ){
		return;
	}
	?>
	<div class="mobile__woocommerce__quick__action__menu">
		<ul>
			<?php
				foreach( $active_quick_actions_links as $single_link => $single_link_data ){
					echo utech_kses($quick_links_data[$single_link_data]);
				}
			?>
		</ul>
	</div>

	<?php
}
if ( '1' == utech_get_option( 'enable_mobile_quick_actions_links', 0 ) ) {
	add_action( 'wp_footer', 'utech_mobile_quick_action_menu' );
}

/*===================================
	SINGLE PRODUCT CUSTOMIZE
=====================================*/
function utech_single_product_page_buy_now_button() {
	global $woocommerce,$product;

	if ( $product->is_type( 'variable' ) && '1' == utech_get_option( 'disable_buy_now_button_in_varable_product', 1 ) ){
		return;
	}

	echo '<a class="buy-now-button button" href="'.home_url('/checkout/?add-to-cart='.get_the_ID()).'">'.utech_get_option( 'buy_now_in_single_product_text', 'Buy Now' ).'</a>';
}
if ( '1' == utech_get_option( 'add_buy_now_in_single_product', 0 ) ) {
	add_action( 'woocommerce_after_add_to_cart_button', 'utech_single_product_page_buy_now_button' );
}